<?php
/*
This is a simple demonstration of the basic concepts.
- View - takes care of rendering
- Model - working with data, or provide access to database
- Presenter - connects View and Model
In Model-View-Presenter, the View and Model can not directly communicate.

We would also split the files into multiple views.

You can try to access this page using query arguments
?pageSize=1
?pageSize=2
?pageSize=2&page=1

In the example below we are using type hinting.
It is optional, unlike in Python this is enforced at runtime.

*/

class Presenter {

  private $view;

  private $model;

  // Here we use type hinting to make sure we get the right instances.
  public function __construct(View $view, Model $model) {
    $this->view = $view;
    $this->model = $model;
  }

  public function handleGet($get) {
    // Validation, here we use filter_var_array which is like filter_input
    // just from a variable, we can also request multiple values at once.
    $arguments = filter_var_array($get, [
      // And a page for pagination
      'page' => [
        'filter' => FILTER_VALIDATE_INT,
        'options' => [
          'min_range' => 0,
          'default' => 0,
        ],
      ],
      'pageSize' => [
        // Page size is between 5 and 20, with 5 the default.
        'filter' => FILTER_VALIDATE_INT,
        'flags'  => FILTER_REQUIRE_SCALAR,
        'options' => [
          'min_range' => 1,
          'max_range' => 20,
          'default' => 5,
        ],
      ],
    ], true); // Add missing as null
    // Working with model and data. Here we just do simple pagination.
    $offset =  $arguments['page'] * $arguments['pageSize'];
    $length = $arguments['pageSize'];
    $customers = array_slice($this->model->getCustomers(), $offset, $length);
    // Render view, we pass the data required for rendering.
    $this->view->render([
      'customers' => $customers,
    ]);
  }

}

class View {

  public function render($arguments) {
    // This should be in a template, use PHP Templator for example.
    // In short this is bad code, just for the demonstration.
    ?> Customers: <ul> <?php
    foreach($arguments['customers'] as $customer) {
      ?><li> <?php print(htmlspecialchars($customer['name']))  ?> </li><?php
    }
    ?> </ul> <?php
  }

}

class Model {

  public function getCustomers() {
    // Again just for demonstration, yet the beauty is, that
    // the rest of the application does not care.
    return [
      [ 'name' => 'Ailish' ],
      [ 'name' => 'Bob' ],
      [ 'name' => 'Petr' ],
    ];
  }

}

$view = new View();
$model = new Model();
$presenter = new Presenter($view, $model);

// Take the global state and make it local using arguments.
$presenter->handleGet($_GET);

/*
Notes:
- We can use interfaces for MVP and then provide default implementation.
  That way can easily have multiple models and pick the right one.
  E.g. one model can use static values, like in the example, another
  can load from JSON, another from SQL database.
- It is common that the Model is replaced with a database access layer.
  Something like class Customers ... so then there is no specific
  model for given Presenter-View pair.
- We would name the classes based on the page it creates.
  E.g. for CustomerList we may have:
  - CustomerListPresenter
  - CustomerListView
  - Customers - as the database access.
*/
