# Markdown documentation link checker

We will use `<filesystem>`, `<format>` `<iostream>`, `<string>`, `<regex>`, and `<ranges>` from the standard library.

```cpp
#include <filesystem>
#include <format>
#include <iostream>
#include <string>
#include <regex>
#include <ranges>
```

The program will take a single argument, the path to the directory containing the markdown files. It will then iterate over all the files in the directory and its subdirectories recursively (following symlinks). For each markdown file, it will check if all the links to files are valid. If not, it will print the file name and the invalid links.

Use `namespace fs = std::filesystem;` to make the code more readable (in the TODO list, we will still use full names).

TODOs:

- define a regular expression for links: `R"(\[[^\]]*\]\(([^)]*)\))"` (R prefix means "raw string", so we don't have to escape the backslashes, it is not related to regular expressions; read it as `R"(<regular expression>)"`)
  - If you know a better regular expression for links in markdown files, feel free to use it; the provided one is a simple one that should work for most cases
- use recursive directory iterator to iterate over all the entries in the directory
- check if the entry is a regular file and if it has a `.md` extension (`entry.is_regular_file()` and `entry.path().extension() == ".md"`)
- open the file (using the path) and read it line by line
- for each line:
  - use `std::sregex_iterator` to find all the links in the line (`std::sregex_iterator()` marks the end)
  - retrieve the string representation of the link from the match: `match.str(1)`
  - if the link points to a http URL (starts with `http://` or `https://`), skip it (assume it is valid as it is not a local file)
  - create a `std::filesystem::path` from the link (`fs::path(link)`)
  - if the link is a relative path (`.is_relative()`):
    - use `entry.path().parent_path()` to get the directory of the markdown file
    - use the `/` operator to concatenate the directory of the markdown file and the relative path
    - convert the result to the normal form (`.lexically_normal()`) to remove `.` and `..` from the path
    - instead of the original `fs::path(link)`, use the result from the previous step in the following steps
  - if the path does not exist (`!std::filesystem::exists(path)` or `!fs::exists(path)`), print the file name and the broken link
    - use `std::format` to format the output: for example, `std::format("{}: {}", entry.path().string(), link_path.string())`

More TODOs:

- use `<chrono>` to measure the execution time of the program; you can use `std::chrono::steady_clock::now()` to get the current time
  - different time points can be subtracted to get the duration between them
  - then, you can use `std::chrono::duration_cast<std::chrono::milliseconds>(duration)` to convert the duration to milliseconds
  - print the duration at the end of the program: `std::cerr << "Execution time: " << duration << std::endl;`
- add line counter to the output of the broken links: `std::format("{}:{}: {}", entry.path().string(), line_number, link_path.string())`
